<?php
// Start output buffering to prevent any output before PDF generation
ob_start();

session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../vendor/autoload.php';

// Check if user has permission to export reports
if (!hasPermission('export_reports') && !$isSuperAdmin) {
    ob_end_clean(); // Clean buffer before redirect
    $_SESSION['error'] = "You don't have permission to export reports.";
    header('Location: reports.php');
    exit;
}

// Get date range from request
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value, setting_binary FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'store_website', 'store_logo', 'currency')");
$stmt->execute();
$store_settings = [];
$store_logo_binary = null;
$store_logo_filename = null;
while ($row = $stmt->fetch()) {
    $store_settings[$row['setting_key']] = $row['setting_value'];
    if ($row['setting_key'] === 'store_logo' && $row['setting_binary']) {
        $store_logo_binary = $row['setting_binary'];
        $store_logo_filename = $row['setting_value'];
    }
}

$storeName = $store_settings['store_name'] ?? 'Store Name';
$storeAddress = $store_settings['store_address'] ?? 'Store Address';
$storePhone = $store_settings['store_phone'] ?? 'Store Phone';
$storeEmail = $store_settings['store_email'] ?? 'Store Email';
$storeWebsite = $store_settings['store_website'] ?? '';
$storeLogo = $store_settings['store_logo'] ?? '';
$currency = $store_settings['currency'] ?? 'Ksh';

// Create temporary logo file if logo exists in database
$tempLogoPath = null;
if ($store_logo_binary && $store_logo_filename) {
    $tempDir = sys_get_temp_dir();
    $tempLogoPath = $tempDir . '/' . uniqid('logo_') . '_' . $store_logo_filename;
    
    // Check if we have required image extensions
    $hasImagick = extension_loaded('imagick');
    $hasGD = extension_loaded('gd');
    
    if ($hasImagick || $hasGD) {
        // We have image processing capabilities
        if (file_put_contents($tempLogoPath, $store_logo_binary) !== false) {
            // File created successfully
            // Check if it's a PNG with alpha channel and convert if needed
            if (strtolower(pathinfo($store_logo_filename, PATHINFO_EXTENSION)) === 'png') {
                try {
                    if ($hasImagick) {
                        // Use Imagick to convert PNG to JPEG (removes alpha channel)
                        $imagick = new Imagick($tempLogoPath);
                        $imagick->setImageBackgroundColor('white');
                        $imagick->setImageAlphaChannel(Imagick::ALPHACHANNEL_REMOVE);
                        $imagick->setImageFormat('jpeg');
                        $imagick->writeImage($tempLogoPath);
                        $imagick->clear();
                        $imagick->destroy();
                    } elseif ($hasGD) {
                        // Use GD to convert PNG to JPEG
                        $image = imagecreatefrompng($tempLogoPath);
                        if ($image !== false) {
                            // Create a white background
                            $width = imagesx($image);
                            $height = imagesy($image);
                            $newImage = imagecreatetruecolor($width, $height);
                            $white = imagecolorallocate($newImage, 255, 255, 255);
                            imagefill($newImage, 0, 0, $white);
                            
                            // Copy the PNG onto the white background
                            imagecopy($newImage, $image, 0, 0, 0, 0, $width, $height);
                            
                            // Save as JPEG
                            imagejpeg($newImage, $tempLogoPath, 90);
                            
                            // Clean up
                            imagedestroy($image);
                            imagedestroy($newImage);
                        }
                    }
                } catch (Exception $e) {
                    // If conversion fails, remove the file and continue without logo
                    if (file_exists($tempLogoPath)) {
                        unlink($tempLogoPath);
                    }
                    $tempLogoPath = null;
                    error_log("Failed to convert PNG logo: " . $e->getMessage());
                }
            }
        } else {
            $tempLogoPath = null; // Reset if failed
        }
    } else {
        // No image processing extensions available, skip logo
        $tempLogoPath = null;
        error_log("No image processing extensions (GD or Imagick) available for logo handling");
    }
}

// Get product performance data with serial numbers
$stmt = $pdo->prepare("
    SELECT 
        p.id,
        p.name,
        p.price,
        p.quantity as current_stock,
        COALESCE(c.name, 'Uncategorized') as category_name,
        COALESCE(SUM(si.quantity), 0) as total_sold,
        COALESCE(SUM(si.total), 0) as total_revenue,
        COALESCE(COUNT(DISTINCT s.id), 0) as transactions_count,
        COALESCE(AVG(si.quantity), 0) as avg_quantity_per_sale,
        GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.serial_number SEPARATOR ', ') as serial_numbers
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    LEFT JOIN sale_items si ON p.id = si.product_id
    LEFT JOIN sales s ON si.sale_id = s.id AND DATE(s.created_at) BETWEEN ? AND ?
    LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id AND psn.status = 'available'
    GROUP BY p.id, p.name, p.price, p.quantity, c.name
    ORDER BY total_sold DESC, p.name ASC
");
$stmt->execute([$start_date, $end_date]);
$products = $stmt->fetchAll();

// Get overall statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(DISTINCT p.id) as total_products,
        SUM(p.quantity) as total_stock_value,
        COUNT(CASE WHEN p.quantity <= 5 THEN 1 END) as low_stock_count,
        COUNT(CASE WHEN p.quantity = 0 THEN 1 END) as out_of_stock_count,
        SUM(CASE WHEN si.sale_id IS NOT NULL THEN si.quantity ELSE 0 END) as total_sold_period,
        SUM(CASE WHEN si.sale_id IS NOT NULL THEN si.total ELSE 0 END) as total_revenue_period
    FROM products p
    LEFT JOIN sale_items si ON p.id = si.product_id
    LEFT JOIN sales s ON si.sale_id = s.id AND DATE(s.created_at) BETWEEN ? AND ?
");
$stmt->execute([$start_date, $end_date]);
$stats = $stmt->fetch();

// Get category performance
$stmt = $pdo->prepare("
    SELECT 
        COALESCE(c.name, 'Uncategorized') as category,
        COUNT(DISTINCT p.id) as products_count,
        SUM(p.quantity) as total_stock,
        COALESCE(SUM(si.quantity), 0) as total_sold,
        COALESCE(SUM(si.total), 0) as total_revenue
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    LEFT JOIN sale_items si ON p.id = si.product_id
    LEFT JOIN sales s ON si.sale_id = s.id AND DATE(s.created_at) BETWEEN ? AND ?
    GROUP BY c.name
    ORDER BY total_revenue DESC
");
$stmt->execute([$start_date, $end_date]);
$categories = $stmt->fetchAll();

// Create PDF using TCPDF
class MYPDF extends TCPDF {
    public function Header() {
        global $storeName, $storeAddress, $storePhone, $storeEmail, $storeWebsite, $tempLogoPath, $start_date, $end_date;
        
        // Report title first
        $this->SetFont('helvetica', 'B', 18);
        $this->Cell(0, 12, 'PRODUCTS PERFORMANCE REPORT', 0, 1, 'C');
        $this->Ln(3);
        
        // Store logo (if exists)
        if (!empty($tempLogoPath) && file_exists($tempLogoPath)) {
            // Get image dimensions and calculate position for centering
            $pageWidth = $this->getPageWidth();
            $margins = $this->getMargins();
            $logoWidth = 30; // Set logo width
            $logoX = ($pageWidth - $logoWidth) / 2; // Center horizontally
            
            $this->Image($tempLogoPath, $logoX, $this->GetY(), $logoWidth, 0, '', '', '', false, 300, '', false, false, 0);
            $this->Ln(20); // Space after logo
        }
        
        // Store information below
        $this->SetFont('helvetica', 'B', 14);
        $this->Cell(0, 8, $storeName, 0, 1, 'C');
        
        $this->SetFont('helvetica', '', 10);
        $this->Cell(0, 5, $storeAddress, 0, 1, 'C');
        
        // Phone and Email on one line
        $this->Cell(0, 5, 'Phone: ' . $storePhone . ' | Email: ' . $storeEmail, 0, 1, 'C');
        
        // Website on separate line if exists
        if (!empty($storeWebsite)) {
            $this->Cell(0, 5, 'Website: ' . $storeWebsite, 0, 1, 'C');
        }
        
        $this->Ln(8);
        
        // Report period after store information
        $this->SetDrawColor(0,0,0);
        $this->Line($this->GetX(), $this->GetY(), $this->getPageWidth() - $this->GetX(), $this->GetY());
        $this->Ln(2);
        $this->SetFont('helvetica', 'B', 12);
        $this->Cell(0, 8, 'Report Period: ' . date('M d, Y', strtotime($start_date)) . ' to ' . date('M d, Y', strtotime($end_date)), 0, 1, 'C');
        $this->Line($this->GetX(), $this->GetY(), $this->getPageWidth() - $this->GetX(), $this->GetY());
        $this->Ln(30);
    }
    
    public function Footer() {
        $this->SetY(-15);
        $this->SetFont('helvetica', 'I', 8);
        $this->Cell(0, 10, 'Generated on ' . date('Y-m-d H:i:s') . ' | Page ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, 0, 'C');
    }
}

// Create new PDF document
$pdf = new MYPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

// Set document information
$pdf->SetCreator('POS System');
$pdf->SetAuthor($storeName);
$pdf->SetTitle('Products Report - ' . $start_date . ' to ' . $end_date);
$pdf->SetSubject('Products Performance Report');

// Set margins (2cm left and right, increased top margin to accommodate larger header)
$pdf->SetMargins(20, 50, 20); // 20mm = 2cm
$pdf->SetHeaderMargin(5);
$pdf->SetFooterMargin(10);

// Set auto page breaks
$pdf->SetAutoPageBreak(TRUE, 25);

// Add a page
$pdf->AddPage();

// Summary statistics
$pdf->SetFont('helvetica', 'B', 11);
$pdf->Cell(0, 8, 'INVENTORY OVERVIEW', 0, 1, 'L');
$pdf->Ln(2);

$pdf->SetFont('helvetica', '', 10);
$pdf->Cell(50, 6, 'Total Products:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['total_products']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Total Stock Units:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['total_stock_value']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Low Stock Items:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['low_stock_count']) . ' (≤5 units)', 0, 1, 'L');

$pdf->Cell(50, 6, 'Out of Stock:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['out_of_stock_count']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Units Sold (Period):', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['total_sold_period']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Revenue (Period):', 0, 0, 'L');
$pdf->Cell(40, 6, $currency . ' ' . number_format($stats['total_revenue_period'], 2), 0, 1, 'L');

$pdf->Ln(10);

// Category performance
$pdf->SetFont('helvetica', 'B', 11);
$pdf->Cell(0, 8, 'CATEGORY PERFORMANCE', 0, 1, 'L');
$pdf->Ln(2);

// Category table header
$pdf->SetFont('helvetica', 'B', 9);
$pdf->SetFillColor(230, 230, 230);
$pdf->Cell(40, 8, 'Category', 1, 0, 'C', true);
$pdf->Cell(25, 8, 'Products', 1, 0, 'C', true);
$pdf->Cell(25, 8, 'Stock', 1, 0, 'C', true);
$pdf->Cell(25, 8, 'Sold', 1, 0, 'C', true);
$pdf->Cell(30, 8, 'Revenue', 1, 0, 'C', true);
$pdf->Cell(25, 8, 'Avg/Product', 1, 1, 'C', true);

// Category data
$pdf->SetFont('helvetica', '', 8);
foreach ($categories as $category) {
    $avg_revenue = $category['products_count'] > 0 ? $category['total_revenue'] / $category['products_count'] : 0;
    
    $pdf->Cell(40, 6, substr($category['category'], 0, 18), 1, 0, 'L');
    $pdf->Cell(25, 6, number_format($category['products_count']), 1, 0, 'C');
    $pdf->Cell(25, 6, number_format($category['total_stock']), 1, 0, 'C');
    $pdf->Cell(25, 6, number_format($category['total_sold']), 1, 0, 'C');
    $pdf->Cell(30, 6, $currency . ' ' . number_format($category['total_revenue'], 2), 1, 0, 'R');
    $pdf->Cell(25, 6, $currency . ' ' . number_format($avg_revenue, 2), 1, 1, 'R');
}

$pdf->Ln(10);

// Products performance table
$pdf->SetFont('helvetica', 'B', 11);
$pdf->Cell(0, 8, 'DETAILED PRODUCT PERFORMANCE', 0, 1, 'L');
$pdf->Ln(2);

// Table header - enlarged to match category performance table width
$pdf->SetFont('helvetica', 'B', 8);
$pdf->SetFillColor(230, 230, 230);
$pdf->Cell(50, 8, 'Product Name', 1, 0, 'C', true);
$pdf->Cell(40, 8, 'Serial Number', 1, 0, 'C', true);
$pdf->Cell(25, 8, 'Price', 1, 0, 'C', true);
$pdf->Cell(20, 8, 'Stock', 1, 0, 'C', true);
$pdf->Cell(20, 8, 'Sold', 1, 0, 'C', true);
$pdf->Cell(15, 8, 'Status', 1, 1, 'C', true);

// Table data
$pdf->SetFont('helvetica', '', 7);

foreach ($products as $product) {
    // Check if we need a new page
    if ($pdf->GetY() > 250) {
        $pdf->AddPage();
        // Re-add table header
        $pdf->SetFont('helvetica', 'B', 8);
        $pdf->SetFillColor(230, 230, 230);
        $pdf->Cell(50, 8, 'Product Name', 1, 0, 'C', true);
        $pdf->Cell(40, 8, 'Serial Number', 1, 0, 'C', true);
        $pdf->Cell(25, 8, 'Price', 1, 0, 'C', true);
        $pdf->Cell(20, 8, 'Stock', 1, 0, 'C', true);
        $pdf->Cell(20, 8, 'Sold', 1, 0, 'C', true);
        $pdf->Cell(15, 8, 'Status', 1, 1, 'C', true);
        $pdf->SetFont('helvetica', '', 7);
    }
    
    // Color code based on stock level
    if ($product['current_stock'] == 0) {
        $pdf->SetFillColor(255, 200, 200); // Red for out of stock
    } elseif ($product['current_stock'] <= 5) {
        $pdf->SetFillColor(255, 255, 200); // Yellow for low stock
    } else {
        $pdf->SetFillColor(255, 255, 255); // White for normal stock
    }
    
    // Format serial numbers for display
    $serialNumbers = $product['serial_numbers'] ?? '';
    $displaySerial = '';
    if (!empty($serialNumbers)) {
        $serials = explode(', ', $serialNumbers);
        if (count($serials) > 3) {
            $displaySerial = $serials[0] . ', ' . $serials[1] . ', ' . $serials[2] . '...';
        } else {
            $displaySerial = substr($serialNumbers, 0, 30);
        }
    } else {
        $displaySerial = 'N/A';
    }
    
    $pdf->Cell(50, 6, substr($product['name'], 0, 30), 1, 0, 'L', true);
    $pdf->Cell(40, 6, $displaySerial, 1, 0, 'L', true);
    $pdf->Cell(25, 6, $currency . ' ' . number_format($product['price'], 2), 1, 0, 'R', true);
    $pdf->Cell(20, 6, number_format($product['current_stock']), 1, 0, 'C', true);
    $pdf->Cell(20, 6, number_format($product['total_sold']), 1, 0, 'C', true);
    $pdf->Cell(15, 6, 'Active', 1, 1, 'C', true);
}

// Add legend
$pdf->Ln(5);
$pdf->SetFont('helvetica', 'B', 9);
$pdf->Cell(0, 6, 'LEGEND:', 0, 1, 'L');
$pdf->SetFont('helvetica', '', 8);

$pdf->SetFillColor(255, 200, 200);
$pdf->Cell(15, 5, '', 1, 0, 'C', true);
$pdf->Cell(30, 5, 'Out of Stock', 0, 0, 'L');

$pdf->SetFillColor(255, 255, 200);
$pdf->Cell(15, 5, '', 1, 0, 'C', true);
$pdf->Cell(30, 5, 'Low Stock (≤5)', 0, 0, 'L');

$pdf->SetFillColor(255, 255, 255);
$pdf->Cell(15, 5, '', 1, 0, 'C', true);
$pdf->Cell(30, 5, 'Normal Stock', 0, 1, 'L');

// Generate filename
$filename = 'Products_Report_' . $start_date . '_to_' . $end_date . '.pdf';

// Create absolute path for reports directory
$reports_dir = dirname(__DIR__) . '/Reports/';
if (!file_exists($reports_dir)) {
    mkdir($reports_dir, 0755, true);
}

// Save PDF to reports folder (optional - for record keeping)
$filepath = $reports_dir . $filename;
try {
    $pdf->Output($filepath, 'F');
} catch (Exception $e) {
    // If saving fails, continue with download only
    error_log("Failed to save PDF to reports folder: " . $e->getMessage());
}

// Clean up temporary logo file
if ($tempLogoPath && file_exists($tempLogoPath)) {
    unlink($tempLogoPath);
}

// Clean any output buffer before sending PDF
ob_end_clean();

// Output PDF to browser for download
$pdf->Output($filename, 'D');
exit;
?> 